﻿namespace Hims.Api.Controllers
{
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [AllowAnonymous]
    [Route("api/problem-list")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ProblemListController : BaseController
    {
        /// <summary>
        /// The icdCode services.
        /// </summary>
        private readonly IProblemListService problemListService;

        /// <inheritdoc />
        public ProblemListController(IProblemListService problemListService) => this.problemListService = problemListService;

        /// <summary>
        /// The fetch async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<ProblemListModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody]ProblemListFilterModel model)
        {
            model = (ProblemListFilterModel)EmptyFilter.Handler(model);
            var icdCodes = await this.problemListService.FetchAsync(model);
            return this.Success(icdCodes);
        }

        /// <summary>
        /// The add async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody]ProblemListModel model)
        {
            model = (ProblemListModel)EmptyFilter.Handler(model);
            var response = await this.problemListService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            return this.Success("Problem list has been added successfully.");
        }

        /// <summary>
        /// The update async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody]ProblemListModel model)
        {
            model = (ProblemListModel)EmptyFilter.Handler(model);
            var response = await this.problemListService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            return this.Success("Problem list has been updated successfully.");
        }

        /// <summary>
        /// The delete async.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody]ProblemListModel model)
        {
            try
            {
                model = (ProblemListModel)EmptyFilter.Handler(model);
                var response = await this.problemListService.DeleteAsync(model.ProblemListId);
                if (response == 0)
                {
                    return this.ServerError();
                }

                return this.Success("Problem list has been deleted successfully.");
            }
            catch (NpgsqlException)
            {
                return this.ServerError();
            }
        }
    }
}